<?php

/**
 * This is the model class for table "Pracownik".
 *
 * The followings are the available columns in table 'Pracownik':
 * @property double $id
 * @property string $imie
 * @property string $nazwisko
 * @property string $pesel
 * @property string $nip
 * @property double $wynagrodzenie
 * @property string $urlAvatara
 * @property double $idAgencji
 * @property string $idDziedziny
 *
 * @property PracownikFirmy $work
 */
class Pracownik extends CActiveRecord
{
	public static $agencyType = array(0=>'Urząd pracy',1=>'Agencja pośrednictwa pracy');
	public $image;
	public $tempFolderName;
	public $tempImageName;
	/**
	 * Returns the static model of the specified AR class.
	 * @param string $className active record class name.
	 * @return Pracownik the static model class
	 */
	public static function model($className=__CLASS__)
	{
		return parent::model($className);
	}

	/**
	 * @return string the associated database table name
	 */
	public function tableName()
	{
		return 'Pracownik';
	}

	/**
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
		return array(
			array('wynagrodzenie, idAgencji, idDziedziny', 'numerical'),
			array('imie, nazwisko', 'length', 'max'=>64),
			array('pesel', 'length', 'max'=>22),
			array('nip', 'length', 'max'=>20),
			array('urlAvatara', 'length', 'max'=>512),
			array('image', 'file','safe' => true, 'types'=>'jpg, gif, png', 'allowEmpty'=>true),
			array('tempFolderName,tempImageName', 'safe'),
			array('nip', 'ext.Validators.IdentityNumberValidator'),
			array('pesel', 'ext.Validators.PeselValidator'),
			array('imie, nazwisko, pesel, nip, wynagrodzenie, idAgencji, idDziedziny', 'required'),

			array('id, imie, nazwisko, pesel, nip, wynagrodzenie, urlAvatara, idAgencji, idDziedziny', 'safe', 'on'=>'search'),
		);
	}

	public function getAvatarUrl()
	{
		if (file_exists($this->urlAvatara))
			return Yii::app()->baseUrl . '/' . $this->urlAvatara;
		else
			return Yii::app()->baseUrl . '/images/gui/empty-avatar.png';
	}

	public function getImieNazwisko()
	{
		return $this->imie . ' ' . $this->nazwisko;
	}

	/**
	 * @return array relational rules.
	 */
	public function relations()
	{
		return array(
			'dziedzinaPracownikas'=>array(self::HAS_MANY, 'DziedzinaPracownika', 'idPracownika'),
			'szkoleniePracownikas'=>array(self::HAS_MANY, 'SzkoleniePracownika', 'idPracownika'),
			'pracownikFirmies'=>array(self::HAS_MANY, 'PracownikFirmy', 'idPracownika'),
			'punkty'=>array(self::HAS_MANY, 'PunktyDziedzinyPracownika', 'idPracownika'),
			'dziedzina'=>array(self::BELONGS_TO, 'Dziedzina', 'idDziedziny'),
			'work'=>array(self::HAS_ONE, 'PracownikFirmy', 'idPracownika',
				'condition' => '"work"."dataZwolnienia" is null'),
			'isWorking'=>array(self::STAT, 'PracownikFirmy', 'idPracownika',
				'condition' => '"t"."dataZwolnienia" is null'),
		);
	}

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array(
			'id'=>Yii::t('Pracownik', 'ID'),
			'imie'=>Yii::t('Pracownik', 'Imię'),
			'nazwisko'=>Yii::t('Pracownik', 'Nazwisko'),
			'pesel'=>Yii::t('Pracownik', 'PESEL'),
			'nip'=>Yii::t('Pracownik', 'Dowód osobisty'),
			'wynagrodzenie'=>Yii::t('Pracownik', 'Wynagrodzenie [zł]'),
			'urlAvatara'=>Yii::t('Pracownik', 'Awatar'),
			'idAgencji'=>Yii::t('Pracownik', 'Typ agencji'),
			'image'=>Yii::t('Pracownik', 'Awatar'),
			'idDziedziny'=>Yii::t('Pracownik', 'Dziedzina'),
		);
	}

	/**
	 * Retrieves a list of models based on the current search/filter conditions.
	 * @return CActiveDataProvider the data provider that can return the models based on the search/filter conditions.
	 */
	public function search()
	{
		$criteria = new CDbCriteria;

		$criteria->compare('"id"', $this->id);
		$criteria->compare('LOWER("imie")', $this->imie, true);
		$criteria->compare('LOWER("nazwisko")', $this->nazwisko, true);
		$criteria->compare('"pesel"', $this->pesel, true);
		$criteria->compare('LOWER("nip")', $this->nip, true);
		$criteria->compare('"wynagrodzenie"', $this->wynagrodzenie);
		$criteria->compare('LOWER("urlAvatara")', $this->urlAvatara, true);
		$criteria->compare('"idAgencji"', $this->getAgencjaNum());
		$criteria->compare('"idDziedziny"', $this->idDziedziny);

		return new CActiveDataProvider($this, array(
			'criteria'=>$criteria,
		));
	}

	public function getAgencjaNum()
	{
		if (empty($this->idAgencji) || is_numeric($this->idAgencji))
			return $this->idAgencji;
		$agencja = strtolower($this->idAgencji);
		foreach (self::$agencyType as $k => $v)
			if (strpos(strtolower($v), $agencja) !== false)
				return $k;
		return -1;
	}

	public function getAgencja()
	{
		return isset(self::$agencyType[$this->idAgencji]) ? self::$agencyType[$this->idAgencji] : '';
	}

	public static function getAvileableWorkers($dziedzinaId, $agencja = 0)
	{
		$pracownicy = Pracownik::model()->with('dziedzina', 'szkoleniePracownikas.idSzkolenia0.dziedzina',
			'dziedzinaPracownikas.idDziedziny0')
		->findAll('"t"."idDziedziny" = :idDziedziny AND "t"."idAgencji" = :idAgencji and
			(select count(*) from "PracownikFirmy" pf where pf."idPracownika" = "t"."id" and pf."dataZwolnienia" is null) = 0', array(
				':idDziedziny' => $dziedzinaId,
				':idAgencji' => $agencja
		));
		$workers = array();
		foreach ($pracownicy as $pracownik)
		{
			$rec = $pracownik->attributes;
			$rec['urlAvatara'] = $pracownik->avatarUrl;
			$rec['szkolenie'] = array();
			foreach($pracownik->szkoleniePracownikas as $szkolenie)
				$rec['szkolenie'][] = $szkolenie->idSzkolenia0->attributes + $szkolenie->attributes
					+ array('dziedzina' => $szkolenie->idSzkolenia0->dziedzina->nazwa);
			$rec['punkty'] = array('Matematyka' => 0, 'Fizyka' => 0, 'Biologia' => 0, 'Chemia' => 0);
			foreach ($pracownik->dziedzinaPracownikas as $punkt)
				$rec['punkty'][$punkt->idDziedziny0->nazwa] += (float)$punkt->punkty;
			$workers[$pracownik->id] = $rec;
		}

		$minLiczbaPracownikow = ParametryGry::getInt('MIN_LICZBA_PRACOWNIKOW', 12);

		if(count($workers) < $minLiczbaPracownikow)
		{
			$workersToGenerate = $minLiczbaPracownikow - count($workers);

			$workersArray = self::generateNewWorkers($dziedzinaId, $agencja, $workersToGenerate);

			return CMap::mergeArray(
				$workers,
				$workersArray
			);
		}
		return $workers;
	}

	private static function generateNewWorkers($dziedzinaId, $agencja, $workersToGenerate)
	{
		$wynagrodzenia = array('100'=>1300,'200'=>1500,'300'=>1700,'400'=>1950,'500'=>2100,'600'=>2350,'700'=>2500,'800'=>2750,'900'=>3150);

		$firstNames = Yii::app()->cache->get('firstNames');
		if($firstNames === false)
		{
			$firstNames = Yii::app()->db->createCommand('select distinct "imie" from "Pracownik"')->queryAll();
			Yii::app()->cache->set('firstNames', $firstNames);
		}

		$lastNamesFemale = Yii::app()->cache->get('lastNamesFemale');
		if($lastNamesFemale === false)
		{
			$lastNamesFemale = Yii::app()->db->createCommand('select distinct "nazwisko" from "Pracownik" where "nazwisko" like \'%a\'')->queryAll();
			Yii::app()->cache->set('lastNamesFemale', $lastNamesFemale);
		}

		$lastNamesMale = Yii::app()->cache->get('lastNamesMale');
		if($lastNamesMale === false)
		{
			$lastNamesMale = Yii::app()->db->createCommand('select distinct "nazwisko" from "Pracownik" where "nazwisko" not like \'%a\'')->queryAll();
			Yii::app()->cache->set('lastNamesMale', $lastNamesMale);
		}

		$peselNips = Yii::app()->cache->get('peselNips');
		if($peselNips === false)
		{
			$peselNips = Yii::app()->db->createCommand('select distinct "pesel", "nip" from "Pracownik"')->queryAll();
			Yii::app()->cache->set('peselNips', $peselNips);
		}

		$avatarsMale = array(
			'1'=>array(
				'images/avatars/matematyka/1.png',
				'images/avatars/matematyka/2.png',
				'images/avatars/matematyka/3.png',
				'images/avatars/matematyka/4.png',
				'images/avatars/matematyka/7.png',
				'images/avatars/matematyka/8.png'
			),
			'2'=>array(
				'images/avatars/fizyka/7.png',
				'images/avatars/fizyka/8.png',
				'images/avatars/dodatkowe/21.jpg',
				'images/avatars/dodatkowe/2.jpg'
			),
			'3'=>array(
				'images/avatars/biologia/4.png',
				'images/avatars/dodatkowe/2.jpg',
				'images/avatars/dodatkowe/5.jpg',
				'images/avatars/dodatkowe/6.jpg',
				'images/avatars/dodatkowe/7.jpg',
				'images/avatars/dodatkowe/8.jpg',
				'images/avatars/dodatkowe/9.jpg'
			),
			'4'=>array(
				'images/avatars/chemia/1.png',
				'images/avatars/chemia/2.png',
				'images/avatars/chemia/3.png',
				'images/avatars/chemia/4.png',
				'images/avatars/chemia/7.png',
				'images/avatars/chemia/8.png'
			)
		);

		$avatarsFemale = array(
			'1'=>array(
				'images/avatars/matematyka/5.png',
				'images/avatars/matematyka/6.png',
				'images/avatars/matematyka/9.png'
			),
			'2'=>array(
				'images/avatars/fizyka/5.png',
				'images/avatars/fizyka/6.png',
				'images/avatars/fizyka/9.png',
				'images/avatars/dodatkowe/16.jpg',
				'images/avatars/dodatkowe/3.jpg'
			),
			'3'=>array(
				'images/avatars/dodatkowe/3.jpg',
				'images/avatars/dodatkowe/12.jpg'
			),
			'4'=>array(
				'images/avatars/chemia/5.png',
				'images/avatars/chemia/6.png',
				'images/avatars/chemia/9.png'
			)
		);

		$workersArray = array();

		for($i = 0; $i < $workersToGenerate; $i++)
		{
			$pracownik = new Pracownik();
			$pracownik->idAgencji = $agencja;
			$pracownik->idDziedziny = $dziedzinaId;

			$points = mt_rand(1, 9) . '00';
			$pracownik->wynagrodzenie = $wynagrodzenia[$points];
			$pracownik->pesel = $peselNips[mt_rand(0, count($peselNips) - 1)]['pesel'];
			$pracownik->nip = $peselNips[mt_rand(0, count($peselNips) - 1)]['nip'];


			$pracownik->imie = $firstNames[mt_rand(0,count($firstNames)-1)]['imie'];
			if($pracownik->imie[strlen($pracownik->imie)-1] == "a")
			{
				$pracownik->nazwisko = $lastNamesFemale[mt_rand(0,count($lastNamesFemale)-1)]['nazwisko'];
				$pracownik->urlAvatara = $avatarsFemale[$dziedzinaId][mt_rand(0,count($avatarsFemale[$dziedzinaId])-1)];
			}
			else
			{
				$pracownik->nazwisko = $lastNamesMale[mt_rand(0,count($lastNamesMale)-1)]['nazwisko'];
				$pracownik->urlAvatara = $avatarsMale[$dziedzinaId][mt_rand(0,count($avatarsMale[$dziedzinaId])-1)];
			}

			$pracownik->save(false);
			DziedzinaPracownika::add($pracownik->id, $dziedzinaId, $points);

			$worker['id'] = $pracownik->id;
			$worker['imie'] = $pracownik->imie;
			$worker['nazwisko'] = $pracownik->nazwisko;
			$worker['pesel'] = $pracownik->pesel;
			$worker['nip'] = $pracownik->nip;
			$worker['wynagrodzenie'] = $pracownik->wynagrodzenie;
			$worker['idAgencji'] = $pracownik->idAgencji;
			$worker['idDziedziny'] = $pracownik->idDziedziny;
			$worker['urlAvatara'] = $pracownik->urlAvatara;

			switch($dziedzinaId)
			{
				case 1:
					$worker['punkty'] = array('Matematyka' => $points, 'Fizyka' => 0, 'Biologia' => 0, 'Chemia' => 0);
					break;
				case 2:
					$worker['punkty'] = array('Matematyka' => 0, 'Fizyka' => $points, 'Biologia' => 0, 'Chemia' => 0);
					break;
				case 3:
					$worker['punkty'] = array('Matematyka' => 0, 'Fizyka' => 0, 'Biologia' => $points, 'Chemia' => 0);
					break;
				case 4:
					$worker['punkty'] = array('Matematyka' => 0, 'Fizyka' => 0, 'Biologia' => 0, 'Chemia' => $points);
					break;
			}

			$workersArray[$worker['id']] = $worker;
		}

		return $workersArray;
	}

	public function sell(Firma &$company)
	{
		static $agencje = array(1 => 147);
		if (empty($agencje[$this->idAgencji]))
			throw new Exception('Nie można odnaleźć agencji.');

		$faktura = new Faktury;
		$faktura->czyKoszt = 1;
		$faktura->idFirmy = $company->id;
		$faktura->idKontrahenta = $agencje[$this->idAgencji];
		$faktura->dataWystawienia = new CDbExpression('SYSDATE');
		$faktura->opis = 'Prowizja za zatrudnienie ' . $this->imie . ' ' . $this->nazwisko;
		$faktura->nrFaktury = 'F/' . date('Y/m/d/', time()) . rand(0, 9) . rand(0, 9) . rand(0, 9);
		$faktura->stawkaVat = ParametryGry::getFloat('PODATEK_VAT', 0.23);
		$faktura->cenaNetto = ParametryGry::getFloat('CENA_APP', 300);
		$company->payOrder($faktura);
		$company->save(false);
		$faktura->save(false);

		return $faktura->id;
	}

	public function getExperience()
	{
		static $experience = array();
		if (!isset($experience[$this->id]))
		{
			$experience[$this->id] = array(
				1 => array(Zlecenie::LEVEL_EASY => 0, Zlecenie::LEVEL_HARD => 0, Zlecenie::LEVEL_VERY_HARD => 0),
				2 => array(Zlecenie::LEVEL_EASY => 0, Zlecenie::LEVEL_HARD => 0, Zlecenie::LEVEL_VERY_HARD => 0),
				3 => array(Zlecenie::LEVEL_EASY => 0, Zlecenie::LEVEL_HARD => 0, Zlecenie::LEVEL_VERY_HARD => 0),
				4 => array(Zlecenie::LEVEL_EASY => 0, Zlecenie::LEVEL_HARD => 0, Zlecenie::LEVEL_VERY_HARD => 0),
			);
			$records = Yii::app()->db->createCommand('
				select z."idDziedziny" "id", z."poziom", count(zf."id") "ilosc" from "ZlecenieFirmy" zf
				left join "Zlecenie" z on z."id" = zf."idZlecenia"
				left join "Dziedzina" d on d."id" = z."idDziedziny"
				where zf."idPracownika" = :workerId and zf."czasZakonczenia" is not null and zf."idOdpowiedzi" is not null
				group by z."idDziedziny", z."poziom"
			')->queryAll(true, array(':workerId' => $this->id));
			foreach ($records as $record)
				$experience[$this->id][(int)$record['id']][(int)$record['poziom']] = (int)$record['ilosc'];
		}
		return $experience[$this->id];
	}

	public function getCertificates()
	{
		static $certs = array();
		if (!isset($certs[$this->id]))
		{
			$certs[$this->id] = array(
				1 => array(Zlecenie::LEVEL_EASY => 0, Zlecenie::LEVEL_HARD => 0, Zlecenie::LEVEL_VERY_HARD => 0),
				2 => array(Zlecenie::LEVEL_EASY => 0, Zlecenie::LEVEL_HARD => 0, Zlecenie::LEVEL_VERY_HARD => 0),
				3 => array(Zlecenie::LEVEL_EASY => 0, Zlecenie::LEVEL_HARD => 0, Zlecenie::LEVEL_VERY_HARD => 0),
				4 => array(Zlecenie::LEVEL_EASY => 0, Zlecenie::LEVEL_HARD => 0, Zlecenie::LEVEL_VERY_HARD => 0),
			);
			$records = Yii::app()->db->createCommand('
				select s."idDziedzina" "id", s."poziom", count(sp."id") "ilosc" from "SzkoleniePracownika" sp
				left join "Szkolenie" s on s."id" = sp."idSzkolenia"
				where sp."idPracownika" = :workerId
				group by s."idDziedzina", s."poziom"
			')->queryAll(true, array(':workerId' => $this->id));
			foreach ($records as $record)
				$certs[$this->id][(int)$record['id']][(int)$record['poziom']] = (int)$record['ilosc'];
		}
		return $certs[$this->id];
	}

	/**
	 * Oszacowuje czas na wykonanie zadania w dniach
	 * @param Zlecenie $task
	 * @return int ilosc sekund na wykonanie
	 */
	public function estimateTaskDuration(Zlecenie &$task)
	{
		$baseDuration = array(
			Zlecenie::LEVEL_EASY => 0.5,
			Zlecenie::LEVEL_HARD => 0.75,
			Zlecenie::LEVEL_VERY_HARD => 1,
		);
		$matrix = array(
			1 => array(1 => 1.00, 2 => 0.50, 3 => 0.25),
			2 => array(1 => 2.00, 2 => 1.00, 3 => 0.50),
			3 => array(1 => 4.00, 2 => 2.00, 3 => 1.00),
		);

		$level = (int)$task->poziom;
		$dziedzina = (int)$task->idDziedziny;

		$base = $baseDuration[$level];

		//max 50%;

		// +/- wynagordzenie +/- 10% lub wiecej (gy różnica w wynagrodzeniu > 25%)
		$minPayment = 1 - ParametryGry::getFloat('AI_MIN_WYNAGRODZENIE', 0.85);
		$paymentP = ParametryGry::getFloat('AI_P_WPLYW_WYNAGRODZENIA', 0.1);
		$paymentDiff = $this->wynagrodzenie - $this->work->wynagrodzenie;
		$base += $paymentP * $paymentDiff / ($this->wynagrodzenie * $minPayment);

		// ++ doświadczenie (dziedzina, poziom) maksmylanie + 20%
		$experience = $this->getExperience();

		$tasksCounts = 0;
		foreach ($experience[$dziedzina] as $lvl => $count)
			$tasksCounts += $matrix[$lvl][$level] * $count;

		$tasksTreshold = ParametryGry::getFloat('AI_P_WPLYW_DOSWIADCZENIA_PROG', 10);

		$tasksCounts /= $tasksTreshold;
		if ($tasksCounts > 1) $tasksCounts = 1;

		$base -= 0.3 * $tasksCounts;

		//odejmij/dodaj z powodu wynagrodzenia
		$certs = $this->getCertificates();

		$certsCounts = 0;
		foreach ($certs[$dziedzina] as $lvl => $count)
			$certsCounts += $matrix[$lvl][$level] * $count;

		$certsTreshold = ParametryGry::getFloat('AI_P_WPLYW_WYKSZTALCENIA_PROG', 5);

		$certsCounts /= $certsTreshold;
		if ($certsCounts > 1) $certsCounts = 1;

		$base -= 0.20 * $certsCounts;

		$period = ParametryGry::getInt('ZLECENIE_ILOSC_DNI', 7);

		$dayPeriod = 24 * 3600;
		$result = round($period * $base * $dayPeriod);
		if ($result < $dayPeriod) $result = $dayPeriod;
		return $result;
	}

	/**
	 * Oszacowuje prawdopodobienstwo wykonania zadania
	 * @param Zlecenie $task
	 * @return float
	 */
	public function estimateTaskSuccessPropability(Zlecenie &$task)
	{
		$basePropability = array(
			Zlecenie::LEVEL_EASY => ParametryGry::getFloat('AI_P_PODSTAWA_LATWY', 0.9),
			Zlecenie::LEVEL_HARD => ParametryGry::getFloat('AI_P_PODSTAWA_TRUDNY', 0.75),
			Zlecenie::LEVEL_VERY_HARD => ParametryGry::getFloat('AI_P_PODSTAWA_BARDZO_TRUDNY', 0.5),
		);
		$matrix = array(
			1 => array(1 => 1.00, 2 => 0.50, 3 => 0.25),
			2 => array(1 => 2.00, 2 => 1.00, 3 => 0.50),
			3 => array(1 => 4.00, 2 => 2.00, 3 => 1.00),
		);

		$level = (int)$task->poziom;
		$dziedzina = (int)$task->idDziedziny;

		//bazowe prawdopodobnieństwo
		$base = $basePropability[$level];

		// +/- wynagordzenie +/- 10% lub wiecej (gy różnica w wynagrodzeniu > 25%)
		$minPayment = 1 - ParametryGry::getFloat('AI_MIN_WYNAGRODZENIE', 0.85);
		$paymentP = ParametryGry::getFloat('AI_P_WPLYW_WYNAGRODZENIA', 0.1);
		$paymentDiff = $this->wynagrodzenie - $this->work->wynagrodzenie;
		$base += $paymentP * $paymentDiff / ($this->wynagrodzenie * $minPayment);

		$p = (1 - $basePropability[$level]) / 2;

		// ++ doświadczenie (dziedzina, poziom) maksmylanie + 20%
		$experience = $this->getExperience();

		$tasksCounts = 0;
		foreach ($experience[$dziedzina] as $lvl => $count)
			$tasksCounts += $matrix[$lvl][$level] * $count;

		$tasksTreshold = ParametryGry::getFloat('AI_P_WPLYW_DOSWIADCZENIA_PROG', 15);

		$tasksCounts /= $tasksTreshold;
		if ($tasksCounts > 1) $tasksCounts = 1;

		$base += $p * $tasksCounts;

		// ++ certyfikaty (dziedzina, poziom)
		$certs = $this->getCertificates();

		$certsCounts = 0;
		foreach ($certs[$dziedzina] as $lvl => $count)
			$certsCounts += $matrix[$lvl][$level] * $count;

		$certsTreshold = ParametryGry::getFloat('AI_P_WPLYW_WYKSZTALCENIA_PROG', 5);

		$certsCounts /= $certsTreshold;
		if ($certsCounts > 1) $certsCounts = 1;

		$base += $p * $certsCounts;

		//clamp value
		if ($base < 0) $base = 0;
		if ($base > 1) $base = 1;

		return $base;
	}

	/**
	 * Sprawia ze pracownik rozwiazuje zadanie
	 * @param ZlecenieFirmy $task zdani do rozwiaznia
	 * @param boolean $forceResult Czy wymusic podanie odpowiedzi
	 * @return int Czy zlecenie zostało pomyślnie/negatuwnie rozwiazane lub nie
	 */
	public function resolveTask(ZlecenieFirmy &$task, $resolve = false)
	{
		$duration = $this->estimateTaskDuration($task->idZlecenia0) + rand(-24 * 3600, +24 * 3600);
		$canResolve = strtotime($task->czasPrzypisania) + $duration < time();

		if ($resolve || $canResolve)
		{
			$task->czasZakonczenia = new CDbExpression('SYSDATE');
			$propability = $canResolve ?
				($this->estimateTaskSuccessPropability($task->idZlecenia0) + rand(-0.05, +0.05)) : 0;
			if ($propability > 1) $propability = 1;
			$success = rand() <= $propability * getrandmax();
			if ($success)
			{
				$answerIndex = rand() % count($task->idZlecenia0->odpowiedziZlecenia);
				$answer = $task->idZlecenia0->odpowiedziZlecenia[ $answerIndex ];

				$task->odpowiedz = $answer->odpowiedz;
				$task->idOdpowiedzi = $answer->id;

				return ZlecenieFirmy::TASK_SUCCESS;
			}
			else {
				$task->odpowiedz = rand(0,1) == 0 ? 'błędna odpowiedź' : null;
				$task->idOdpowiedzi = null;

				return ZlecenieFirmy::TASK_FAILED;
			}
		}

		return ZlecenieFirmy::TASK_NOTSET;
	}

	public function getErrandEstimation(ZlecenieFirmy &$task)
	{
		$now = time();
		$time = strtotime($task->czasPrzypisania);

		$duration = round(($time + $this->estimateTaskDuration($task->idZlecenia0) - $now) / 24 / 3600);

		return array(
			'duration' => $duration,
			'durationUnit' => StringUtils::numberUnitWorld($duration, array('dzień', 'dni', 'dni')),
			'taskEnd' => round(($task->expireationTime - $now) / 24 / 3600),
			'propability' => $this->estimateTaskSuccessPropability($task->idZlecenia0)
		);
	}

	public function getTaskEstimation(Zlecenie &$task)
	{
		$duration = round($this->estimateTaskDuration($task) / 24 / 3600);
		return array(
			'duration' => $duration,
			'durationUnit' => $duration == 1 ? 'dzień' : 'dni',
			'taskEnd' => ParametryGry::getInt('ZLECENIE_ILOSC_DNI', 7),
			'propability' => $this->estimateTaskSuccessPropability($task)
		);
	}

	public function getMinPayment()
	{
		$minPaymentP = (float)ParametryGry::getValue('AI_MIN_WYNAGRODZENIE', 0.85);
		return $minPaymentP * (float)$this->wynagrodzenie;
	}

	public function barterPayment($payment)
	{
		$workerPayment = (float)$this->wynagrodzenie;
		$minPayment = $this->getMinPayment(); //$this->minPayment;
		if ($payment < $minPayment)
			return 0;
		else if ($payment >= $workerPayment)
			return 1;
		else {
			$p = ($payment - $minPayment) / ($workerPayment - $minPayment);
			return rand() < $p * getrandmax();
		}
	}
}
